/*
Projekt:	R-Bus-Modul8
Autor:		Jrg Plitz
Datei:		R-Bus-Modul8.c
*/
/*----------------------------------------------------------------------------*/

/*==============================================================================
Software fr ein selbstentwickeltes Modul zum Einlesen von 8-Bit 
Rckmeldeinformationen ber den R-Bus einer z(Z)21.
Die Programmierung der Modul-Adresse muss mit dem Z21 Maintenance Tool erfolgen.
==============================================================================*/

#include <avr/io.h>
#include <avr/interrupt.h>

#include "TypeDef.h"
#include "R-Bus-Modul8.h"
#include "Timer.h"
#include "Uart.h"
#include "Eeprom.h"
//------------------------------------------------------------------------------
u8 bModuleAddress;	//Modul-Adresse 1..20
u8 bFeedbackState;
u8 bShowModuleAddressTimer;
u8 bModuleOnlineTimer;
//------------------------------------------------------------------------------
void IOInit(void);
void ClockInit(void);
void ReadFeedback(void);
void SetLED(u8);
void InputPullUpEnable(void);
//------------------------------------------------------------------------------

/*============================================================================*/
int main(void)
{
	ClockInit();	//Taktquelle interner 20MHz geteilt durch 2
	IOInit();
	InputPullUpEnable();
	TimerB0Init();   
	asm("nop");
	
	sei();
	//--------------------------------------------------------------------------
	//Moduladresse aus EEPROM lesen
	bModuleAddress = ReadModuleAddr();
	if ((bModuleAddress > 20) || (bModuleAddress == 0))
	{
		bModuleAddress = 1;	//default Adresse 1
		WriteModuleAddr(bModuleAddress);
	}
	//--------------------------------------------------------------------------
	UartInit();	//Kommunikation starten (Schnittstelle initialisieren)
	//--------------------------------------------------------------------------
	//ein mal kurz alle LEDs an
	SetLED(0xFF);
	while (bTimer1s == 0);
	bTimer1s = 0;
	//--------------------------------------------------------------------------
	//dann kurz die Moduladresse anzeigen
	SetLED(bModuleAddress);
	while (bTimer1s == 0);
	bTimer1s = 0;
	//--------------------------------------------------------------------------
    while (1)	//Beginn Haupt-/Enlosschleife
    {
		//----------------------------------------------------------------------
		if (bTimer10ms)
		{
			bTimer10ms = 0;
			ReadFeedback();	//Zustand der Eingnge einlesen und entprellen
		}
		//----------------------------------------------------------------------
		if (bTimer100ms)
		{
			bTimer100ms = 0;
			//------------------------------------------------------------------
			if (bShowModuleAddressTimer) bShowModuleAddressTimer--;
			else SetLED(bFeedbackState);
			//------------------------------------------------------------------
			if (bModuleOnlineTimer) bModuleOnlineTimer--;
			//------------------------------------------------------------------
		}
		//----------------------------------------------------------------------
		if (bTimer1s)
		{
			bTimer1s = 0;
			//bFeedbackState++;
			//Uart0Sendbyte(0x55);
		}
		//----------------------------------------------------------------------
 		if (bModuleOnlineTimer)
		 {
			if (bBlink & (1<<xBlinkSlow)) mLEDOn;
			else mLEDOff;
		 }
		 else
		 {
			if (bBlink & (1<<xBlinkFast)) mLEDOn;
			else mLEDOff;
		 }
    }
}
/*============================================================================*/

/*============================================================================*/
void IOInit()
{	
	//PA0..3 = LEDA3..LEDB4 -> Ausgnge L-Pegel
	PORTA.OUTCLR = PIN0_bm|PIN1_bm|PIN2_bm|PIN3_bm;	//L-Pegel
	PORTA.DIRSET = PIN0_bm|PIN1_bm|PIN2_bm|PIN3_bm;	//Ausgnge
	//PA4..7 = INB4..IN3A -> Eingang ohne PullUp
	PORTA.DIRCLR = PIN4_bm|PIN5_bm|PIN6_bm|PIN7_bm;	//Eingnge
	PORTA.PIN4CTRL = 0;
	PORTA.PIN5CTRL = 0;
	PORTA.PIN6CTRL = 0;
	PORTA.PIN7CTRL = 0;
	
	//PC0..3 = INB2..INA1 -> Eingang ohne PullUp
	PORTC.DIRCLR = PIN0_bm|PIN1_bm|PIN2_bm|PIN3_bm;	//Eingnge
	PORTC.PIN0CTRL = 0;	//ohne PullUp
	PORTC.PIN1CTRL = 0;
	PORTC.PIN2CTRL = 0;
	PORTC.PIN3CTRL = 0; 
	
	//PD0..3 = LEDA1..LEDB2 -> Ausgnge L-Pegel
	PORTD.OUTCLR = PIN0_bm|PIN1_bm|PIN2_bm|PIN3_bm;	//L-Pegel
	PORTD.DIRSET = PIN0_bm|PIN1_bm|PIN2_bm|PIN3_bm;	//Ausgnge
	//PD4..6 = TXD, RXD, DIR -> UartInit()
	//PD7 = LED -> Ausgang L-Pegel
	PORTD.OUTCLR = PIN7_bm;	//L-Pegel
	PORTD.DIRSET = PIN7_bm;	//Ausgang
	
	//PF0 = PULL -> Eingang (hochohmig, keine externen PullUp fr Belegtmelder)
	PORTF.DIRCLR = PIN0_bm;	//Eingang
	PORTF.PIN0CTRL = 0;	//kein PullUp
//	//PF0 = PULL -> Ausgang H-Pegel (PullUp fr Belegtmelder)
// 	PORTF.OUTSET = PIN0_bm;	//H-Pegel
// 	PORTF.DIRSET = PIN0_bm;	//Ausgang
// 	//PF0 = PULL -> Ausgang L-Pegel (Pulldown fr Weichenrckmelder)
// 	PORTF.OUTCLR = PIN0_bm;	//L-Pegel
// 	PORTF.DIRSET = PIN0_bm;	//Ausgang
	//PF1 = frei -> Eingang mit PullUp
	PORTF.DIRCLR = PIN1_bm;	//Eingang
	PORTF.PIN1CTRL = PORT_PULLUPEN_bm;	//mit PullUp
	//PF6 = RESET
}
/*============================================================================*/

/*============================================================================*/
void ReadFeedback()
{
	u8 bTemp = 0;
	static u8 bFeedbackCnt1, bFeedbackCnt2;

	//Eingnge von PA4..7 und PC0..3 einlesen (Reihenfolge passt nicht)
	//L-aktiv mit internen und externen PullUps fr Kontakte gegen GND
	//L am Eingang = Bit gesetzt
	if ((PORTC.IN & PIN3_bm) == 0) bTemp |= PIN0_bm;	//PC3 = INA1 = Bit0
	if ((PORTC.IN & PIN2_bm) == 0) bTemp |= PIN1_bm;	//PC2 = INB1 = Bit1
	if ((PORTC.IN & PIN1_bm) == 0) bTemp |= PIN2_bm;	//PC1 = INA2 = Bit2
	if ((PORTC.IN & PIN0_bm) == 0) bTemp |= PIN3_bm;	//PC0 = INB2 = Bit3
	if ((PORTA.IN & PIN7_bm) == 0) bTemp |= PIN4_bm;	//PA7 = INA3 = Bit4
	if ((PORTA.IN & PIN6_bm) == 0) bTemp |= PIN5_bm;	//PA6 = INB3 = Bit5
	if ((PORTA.IN & PIN5_bm) == 0) bTemp |= PIN6_bm;	//PA5 = INA4 = Bit6
	if ((PORTA.IN & PIN4_bm) == 0) bTemp |= PIN7_bm;	//PA4 = INB4 = Bit7
	
	//Eingangsinformationen entprellen
	//das Prinzip ist hier wieder von Peter Danneger abgekupfert
	//in bTemp ist fr jeden gerade aktiven Eingang ein Bit gesetzt
	bTemp = bFeedbackState ^ bTemp;   //input changed ?
	bFeedbackCnt1 = ~ (bFeedbackCnt1 & bTemp );      // reset or count ct1
	bFeedbackCnt2 = bFeedbackCnt1 ^ (bFeedbackCnt2 & bTemp);   // reset or count ct2
	bTemp &= bFeedbackCnt1 & bFeedbackCnt2;      // count until roll over
	bFeedbackState ^= bTemp;			// then toggle debounced state
	//hier knnte man noch Flags fr den bergang 0->1 und 1->0 ableiten
	//--------------------------------------------------------------------------
}
/*============================================================================*/

void ClockInit()
{
	CLKCTRL.MCLKCTRLA = 0;	//Internal high-frequency oscillator, kein CLKOUT
 	CCP = CCP_IOREG_gc;		//Register-Sicherheit abschalten
//	CLKCTRL.MCLKCTRLB = CLKCTRL_PDIV_DIV2_gc|CLKCTRL_PEN_bm;
	CLKCTRL.MCLKCTRLB = CLKCTRL_PDIV_DIV4_gc|CLKCTRL_PEN_bm;
}

void SetLED(u8 bLED)
{
	//Ausgnge fr die Kontroll-LEDs an PA0..3 und PD0..3 schalten
	//(die Reihenfolge passt auch hier nicht ganz, deshalb per Nibble)
	//Bit gesetzt = Eingang aktiv = LED leuchten = H-Pegel
	//Bit0..3 = LEDA1..LEDB2 = PD0..3
	//Bit4..7 = LEDA3..LEDB4 = PA0..3
	PORTD.OUTSET = (bLED & 0x0F);
	PORTD.OUTCLR = ~(bLED) & 0x0F;
	PORTA.OUTSET = ((bLED >> 4) & 0x0F);
	PORTA.OUTCLR = ~(bLED >> 4) & 0x0F;
}

void InputPullUpEnable()
{
		//PA4..7 = INB4..IN3A
		PORTA.PIN4CTRL = PORT_PULLUPEN_bm;
		PORTA.PIN5CTRL = PORT_PULLUPEN_bm;
		PORTA.PIN6CTRL = PORT_PULLUPEN_bm;
		PORTA.PIN7CTRL = PORT_PULLUPEN_bm;
		
		//PC0..3 = INB2..INA1
		PORTC.PIN0CTRL = PORT_PULLUPEN_bm;
		PORTC.PIN1CTRL = PORT_PULLUPEN_bm;
		PORTC.PIN2CTRL = PORT_PULLUPEN_bm;
		PORTC.PIN3CTRL = PORT_PULLUPEN_bm;
}