/*
Projekt:	R-Bus-Modul8
Autor:		Jrg Plitz
Datei:		Uart.c
*/
/*----------------------------------------------------------------------------*/

#include <avr/io.h>
#include <avr/interrupt.h>

#include "TypeDef.h"
#include "Uart.h"
#include "R-Bus-Modul8.h"
#include "Eeprom.h"

volatile u8 bUartRxCharCnt;
u8 bRBAddress;

/*==============================================================================
Funktion UartInit()
initialisiert die UART fr die Kommunikation per XPressNet (RS485 Halbduplex, 
62,5kbit/s, 9 Datenbits, keine Paritt, 1 Stoppbit)
==============================================================================*/
void UartInit()
{
	//Uart-IO-Pins PD4..6 konfigurieren
	PORTD.OUTSET = PIN4_bm;					//PD4 (TXD) auf 1
	PORTD.OUTCLR = PIN6_bm;					//PD6 (DIR) auf 0
	PORTD.PIN5CTRL = PORT_PULLUPEN_bm;	//fr PD5 (RXD) PullUp einschalten
	PORTD.DIRCLR = PIN5_bm;					//PD5 (RXD) als Eingang
	PORTD.DIRSET = PIN4_bm;					//PD4 (TXD) als Ausgang
	PORTD.DIRSET = PIN6_bm;					//PD6 (DIR) als Ausgang
	
	//Die IOs der UART noch auf die alternativen Anschlsse PD4..5 verschieben
	PORTMUX.USARTROUTEA = 0x03;		//ALT3 = PD4..5
	
	//62,5kBit/s mit 9N1 einstellen
	USART0.BAUD = kCPUClock * 4 / 62500;
	//nur den Empfangsinterrupt einschalten, keine HW-RS485-Steuerung
	USART0.CTRLA = USART_RXCIE_bm;
	//Sender und Empfnger einschalten, kein Multiprozessor, Normal-Speed
	USART0.CTRLB = USART_RXEN_bm|USART_TXEN_bm|USART_RXMODE_NORMAL_gc;
	//Asyncrone UART, ohne Paritt, 9 Datenbits (TXDATAH muss zuerst geschrieben werden)
	USART0.CTRLC = USART_CMODE_ASYNCHRONOUS_gc|USART_PMODE_DISABLED_gc|USART_SBMODE_1BIT_gc|USART_CHSIZE_9BITH_gc;
	//USART0.CTRLC = USART_CMODE_ASYNCHRONOUS_gc|USART_PMODE_DISABLED_gc|USART_SBMODE_1BIT_gc|USART_CHSIZE_8BIT_gc;
	USART0.CTRLD = 0;	//keine ABW
}
/*============================================================================*/


/*==============================================================================
Interruptserviceroutine USART_RX
Wird ausgelst, wenn ein Byte ber UART empfangen wurde.
Hier werden die Zeichen gleich "vorausgewertet".
Empfang und Auswertung Roco Rckmelde-Bus-Kommunikation
- als CallByte kommt nur 0x1DE vor
- das Headerbyte ist 0xFX (X = Anzahl der folgenden Zeichen)
- als folgendes Infobyte sind fr die normale Rckmeldung nur 0x00 (Gruppe 0)
  oder 0x10 (Gruppe 1) zulssig (0b110G AAAA ist fr die Programmierung der 
  Moduladresse)
- danach legt jeder Rckmelder sein Datenbyte mit der Position entsprechend
  seiner eingestellten Adresse auf den Bus
- am Ende kommt vom Master noch eine Checksumme (die drfte fr die einfache
  Rckmeldung fr den Slave uninteressant sein, er hat ja seine Daten 
  abgeliefert)
- Die Programmierung der Moduladresse ber das PC-Tool erfolgt mit einem 
  Infobyte 0b110G AAAA (-> & 0xE0 == 0xC0). Damit ergeben sich fr die
  Moduladresse 1..10 -> 0xC1..0xCA und fr die Moduladressen 11..20 ->
  0xD1..0xDA. Ungltige Einstellungen werden zwar vom PC-Programm ausgeschlossen
  (Auswahlmglichkeit nur 1..20), es ist aber trotzdem eine Prfung eingebaut.
  Im Ablauf der Programmierung wird vom PC-Programm z.B ein Telegramm
  0x1DE 0x0F1 0x0C6 0xC8 (Adresse 6 programmieren) oder
  0x1DE 0x0F1 0x0D3 0xDD (Adresse 16 programmieren) in schneller Folge (ca. 
  aller 7,3ms) gesendet, bis der Dialog wieder geschlossen wird. Damit sollte
  vor dem Programmieren in den Eeprom geprft werden, ob die neue Adresse schon
  bernommen wurde (gilt auch fr die Anzeige).  
- Die an einer Stelle ("Feedback interface Roco (R Bus/10787)" ebenfalls 
  beschriebene Mglichkeit, die Softwareversion abzufragen, wird hier nicht
  untersttzt. Es gibt auch im PC-Tool keine Anzeige dafr.
  (Info-Byte 0b100G 0000 Software version request)
==============================================================================*/
ISR(USART0_RXC_vect)
{
	u8 bRxDataLow;
	u8 bRxDataHigh;
	u8 bNewAddress;
	
	bRxDataHigh = USART0.RXDATAH;	//Status-Bits und Bit9 einlesen
	bRxDataLow = USART0.RXDATAL;
	

	if ((bRxDataHigh & (USART_FERR_bm|USART_BUFOVF_bm)) == 0)	//Byte i.O. (Rahmen)
	{
		//----------------------------------------------------------------------
		if (bRxDataHigh & (USART_DATA8_bm))
		{	//jedes Callbyte (Bit9 = 1) markiert den Anfang eines Telegramms
			//CallByte (Bit9 = 1)
			if (bRxDataLow == 0xDE)	//-> weitere Daten empfangen
			{
				bUartRxCharCnt = 1;		//weitere Daten empfangen
				bRBAddress = 0;
			}
		}
		//----------------------------------------------------------------------
		else //Bit9 = 0 -> Datenbyte
		{
			if (bUartRxCharCnt)	//"zweites und alle weiteren empfangenen Bytes"
			{
				if (bUartRxCharCnt == 1)	//das erste Datenbyte ist der Header
				{
					//Header = 0xFC -> 0x0C = Anzahl der folgenden Datenbyte (10 + Group + Xor)
					if ((bRxDataLow & 0xF0) == 0xF0) bUartRxCharCnt++;
					else bUartRxCharCnt = 0;	//als erstes Zeichen nach dem Callbyte kein Header
				}	//Ende Header-Byte
				else if (bUartRxCharCnt == 2)	//das zweite Datenbyte enthlt die Gruppenzuordnung
				{
					if (bRxDataLow == 0x00)		//Gruppe 0
					{
						bRBAddress = 1;
						bUartRxCharCnt++;
					}
					else if (bRxDataLow == 0x10)	//Gruppe1
					{
						bRBAddress = 11;
						bUartRxCharCnt++;
					}
					else if ((bRxDataLow & 0xE0) == 0xC0)	//Programmierung Moduladresse
					{
						//die zu programmierende Moduladresse steht im unteren Nibble, die Gruppe
						//in Bit4
						if (bRxDataLow & 0x0F)	//ungleich 0
						{
							if ((bRxDataLow & 0x0F) <= 10)	//kleiner 10
							{
								bNewAddress = (bRxDataLow & 0x0F);	
								if (bRxDataLow & 0x10) bNewAddress += 10;	//Gruppe 2 = 11-20
								if (bNewAddress != bModuleAddress)
								{
									bModuleAddress = bNewAddress;
									WriteModuleAddr(bModuleAddress);	//Moduladresse merken
								}	//Ende neue Adresse
								SetLED(bModuleAddress);	//Adresse ber alle LEDs anzeigen
								bShowModuleAddressTimer = 10;
							}	//Ende Adress-Nibble < = 10
						}	//Ende Adresse != 0
						bUartRxCharCnt = 0;
					}	//Ende Programmierung Moduladresse
					else if  ((bRxDataLow & 0xE0) == 0x80)	//Abfrage Softwareversion
					{
						bUartRxCharCnt = 0;
					}
					else bUartRxCharCnt = 0;	//als zweites Zeichen nach dem Callbyte keine Gruppe oder Programmierung
				}
				//Bei jedem empfangenem Zeichen, egal ob vom Master oder von
				//einem anderen Slave prfen, ob die Gruppenzuordnung schon
				//erfolgt ist (bRBAddress != 0) und ob die von 1 (Gruppe 0) oder
				//11 (Gruppe 1) weitergezhlte bRBAdrress der eigenen
				//Moduladresse 1..20 entspricht. Dann muss hier direkt die
				//Ausgabe des Rckmeldebytes erfolgen.
				if (bRBAddress != 0)
				{
					if (bRBAddress == bModuleAddress)
					{	//eigene Rckmeldedaten senden (im UDR-IRQ)
						mRS485Tx;	//RS485 auf Senden schalten
						bModuleOnlineTimer = kModuleOfflineTimeout;
						//Aussendung starten (UDRIE freigeben)
						USART0.CTRLA |= USART_DREIE_bm;
					}
					bRBAddress++;	//Moduladresse im Telegramm hochzhlen
				}	//Ende "Gruppenzuordnung ist erfolgt"
			}	//Ende "zweites und alle weiteren empfangenen Bytes"
		}	//Ende "RB8 = 0"
	}	//Ende Byte i.O. (Rahmen)
}
/*============================================================================*/

/*==============================================================================
Interruptserviceroutine USART_DRE
==============================================================================*/
ISR(USART0_DRE_vect)
{
	USART0.TXDATAH = 0;	//immer mit gelschtem 9. Bit
	USART0.TXDATAL = bFeedbackState;
	USART0.CTRLA &= ~(USART_DREIE_bm);	//eigenen IRQ wieder sperren
	USART0.CTRLA |= USART_TXCIE_bm;	//TXC-IRG freigeben
}
/*============================================================================*/

/*==============================================================================
Interruptserviceroutine USART_TX
Wird ausgelst, wenn das (letzte) Zeichen vollstndig ausgesendet worden ist.
Relevant nur fr RS485-Halbduplex Betrieb zum Zurckschalten des Treiber-IC
auf Empfangen.
==============================================================================*/
ISR(USART0_TXC_vect)
{
	USART0.CTRLA &= ~(USART_TXCIE_bm);	//TXC-IRG wieder sperren
	USART0.STATUS |= USART_TXCIF_bm;	//TXCIF-Flag wieder lschen
	mRS485Rx;		//auf Empfangen zurckschalten
}
/*============================================================================*/

void Uart0Sendbyte(u8 bData)
{
	while (!(USART0.STATUS & USART_DREIF_bm));
	mRS485Tx;
	if (bFeedbackState) USART0.TXDATAH = USART_DATA8_bm;
	else USART0.TXDATAH = 0;
	USART0.TXDATAL = bData;
	USART0.STATUS |= USART_TXCIF_bm;
	while (!(USART0.STATUS & USART_TXCIF_bm));
	mRS485Rx;
}